<?php

namespace Workdo\Mail2Ticket\Services;

use App\Events\CreateTicket;
use App\Events\CreateTicketFromThirdParty;
use App\Models\Conversion;
use App\Models\Ticket;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use League\OAuth2\Client\Provider\GenericProvider;
use Webklex\PHPIMAP\ClientManager;
use Workdo\Mail2Ticket\Entities\ImapSetting;

class ImapService
{
    protected $settings;

    protected $client;

    public function __construct(array $settings)
    {
        $this->settings = $settings;
        $this->initializeMailbox();
    }

    protected function initializeMailbox()
    {
        $cacheKey = 'imap_connection_config';

        Log::info($this->settings['mail_type']);
        if ($this->settings['mail_type'] == 'outlook') {
            if (! $this->settings['outlook_access_token'] || ! $this->settings['outlook_refresh_token']) {
                return response('No access token or refresh token available. Please authenticate via OAuth.', 400);
            }
        }
        // Check if IMAP extension is loaded
        if (! function_exists('imap_open')) {
            return response('PHP IMAP extension is not installed or enabled.', 500);
        }
        if ($this->settings['mail_type'] == 'outlook') {
            $config = [
                'host' => $this->settings['outlook_imap_mail_host'],
                'port' => $this->settings['outlook_imap_mail_port'],
                'encryption' => $this->settings['outlook_imap_mail_encryption'],
                'username' => $this->settings['outlook_imap_email_address'],
                'password' => $this->settings['outlook_access_token'], // Use OAuth access token
                'protocol' => $this->settings['outlook_imap_mail_driver'],
                'authentication' => $this->settings['outlook_imap_mail_authentication'],
            ];
        } else {
            $config = [
                'host' => $this->settings['imap_mail_host'] ?? 'imap.gmail.com',
                'port' => $this->settings['imap_mail_port'] ?? 993,
                'encryption' => $this->settings['imap_mail_encryption'] ?? 'ssl',
                'username' => $this->settings['imap_email_address'] ?? '',
                'password' => $this->settings['imap_email_password'] ?? '',
                'protocol' => $this->settings['imap_mail_driver'],
            ];
        }

        try {
            $cm = new ClientManager;
            $this->client = $cm->make($config);
        } catch (Exception $e) {
            Log::error('Failed to initialize IMAP client: '.$e->getMessage());
            throw $e;
        }
    }

    protected function extractReplyText($body, $isReply = false)
    {
        if ($isReply) {
            $patterns = [
                '/On\s.+?wrote:/is',
                '/From:\s.+?\nSent:.+/is',
                '/^\s*>.+$/m',
                '/^\s*-----Original Message-----.*$/ism',
                '/On\s.+?at\s.+?wrote:/is',
                '/^.*(\d{1,2}:\d{2}\s*(AM|PM)).+wrote:$/m',
            ];

            foreach ($patterns as $pattern) {
                if (preg_match($pattern, $body, $matches, PREG_OFFSET_CAPTURE)) {
                    $body = substr($body, 0, $matches[0][1]);
                    break;
                }
            }
        }

        return trim($body);
    }

    protected function refreshAccessToken($settings)
    {
        $provider = new GenericProvider([
            'clientId' => $settings['outlook_client_id'] ?? '',
            'clientSecret' => $settings['outlook_client_secret'] ?? '',
            'redirectUri' => env('APP_URL').'outlook/webhook',
            'urlAuthorize' => 'https://login.microsoftonline.com/consumers/oauth2/v2.0/authorize',
            'urlAccessToken' => 'https://login.microsoftonline.com/consumers/oauth2/v2.0/token',
            'urlResourceOwnerDetails' => '',
            'scopes' => 'https://outlook.office.com/IMAP.AccessAsUser.All offline_access',
        ]);

        try {
            $newToken = $provider->getAccessToken('refresh_token', [
                'refresh_token' => $settings['outlook_refresh_token'] ?? '',
            ]);

            $accessToken = $newToken->getToken();
            $refreshToken = $newToken->getRefreshToken() ?? $settings['outlook_refresh_token'];
            $expiresAtTimestamp = $newToken->getExpires();
            $expiresAt = date('Y-m-d H:i:s', $expiresAtTimestamp);

            $data = [
                'outlook_access_token' => $accessToken,
                'outlook_refresh_token' => $refreshToken,
                'outlook_expires_at' => $expiresAt,
            ];

            foreach ($data as $key => $value) {
                ImapSetting::updateOrInsert(
                    ['name' => $key],
                    [
                        'value' => $value,
                        'created_by' => 1,
                    ]
                );
            }
            if (Cache::get('imap_connection_config')) {
                Cache::flush('imap_connection_config');
            }
            Log::info('Access token refreshed successfully.');
        } catch (\Exception $e) {
            Log::error('Error refreshing token: '.$e->getMessage());
            throw new Exception('Failed to refresh access token.');
        }
    }

    public function fetchUnreadEmails($settings)
    {
        if (! $this->client) {
            Log::error('IMAP client not initialized.');

            return [];
        }
        $expiresAt = $settings['outlook_expires_at'] ?? null;
        $expiresAtTimestamp = $expiresAt ? strtotime($expiresAt) : 0;

        // When Refresh Token Expired Generate New Token Automatically.
        $now = time();
        if (($expiresAtTimestamp <= $now + 300) && (isset($settings['mail_type']) && $settings['mail_type'] == 'outlook')) {
            $this->refreshAccessToken($settings);
            $settings = ImapSetting::pluck('value', 'name')->toArray();
        }

        try {
            $this->client->connect();
            Log::info('Connecting to IMAP server...');

            // Fetch the INBOX folder
            if ($this->settings['mail_type'] == 'outlook') {
                $folder = $this->client->getFolder('Inbox');
            } else {
                $folder = $this->client->getFolder('INBOX');
            }
            if ($folder === null) {
                Log::error('INBOX folder not found.');

                return [];
            }
            Log::info('Successfully fetched INBOX folder.');

            // Search for unread messages
            $unreadMessages = $folder->messages()->where('UNSEEN')->limit(10)->get();
            Log::info('Found '.$unreadMessages->count().' unread messages.');

            if ($unreadMessages->isEmpty()) {
                Log::info('No unread messages found.');

                return [];
            }

            $emailRecords = [];

            foreach ($unreadMessages as $message) {
                $rawBody = $message->getTextBody() ?? '';
                $isReply = $message->getSubject() && Str::startsWith($message->getSubject(), 'Re:');

                $uniqueSubject = ! $isReply ? 'Re: '.time().' '.($message->getSubject() ?? 'No Subject') : ($message->getSubject() ?? 'No Subject');
                $date = $message->getDate();
                $receivedAt = $date && method_exists($date, 'format') ? $date->format('Y-m-d H:i:s') : ((string) $date ?: now()->format('Y-m-d H:i:s'));

                $emailRecords[] = [
                    'message_id' => $message->getMessageId(),
                    'from' => $message->getFrom()[0]->mail ?? 'Unknown',
                    'subject' => $uniqueSubject,
                    'body' => $this->extractReplyText($rawBody, $isReply) ?? '',
                    'received_at' => $receivedAt,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];

                $existingTicket = Ticket::where('subject', 'LIKE', '%'.$uniqueSubject)->latest()->first();
                if ($existingTicket) {
                    $conversion = new Conversion;
                    $conversion->ticket_id = $existingTicket->id;
                    $conversion->description = $this->extractReplyText($rawBody, $isReply) ?? '';
                    $conversion->sender = 'user';
                    $conversion->save();
                    manageFrontToAdminPusher($conversion, $existingTicket);
                } else {
                    $baseTicketId = time();
                    $attempt = 0;
                    $ticketId = $baseTicketId;

                    while (Ticket::where('ticket_id', $ticketId)->exists()) {
                        $ticketId = $baseTicketId + $attempt;
                        $attempt++;
                        if ($attempt > 9) {
                            Log::error('Unable to generate unique ticket_id after 10 attempts for message ID: '.$message->getMessageId());
                            throw new Exception('Unable to generate unique ticket_id after 10 attempts');
                        }
                    }
                    $ticket = new Ticket;
                    $ticket->ticket_id = $ticketId;
                    $ticket->name = $message->getFrom()[0]->personal ?? $message->getFrom()[0]->mail ?? 'Unknown';
                    $ticket->email = $message->getFrom()[0]->mail ?? 'Unknown';
                    $ticket->subject = $uniqueSubject;
                    $ticket->status = 'New Ticket';
                    $ticket->is_ticket_assign_to_agent = 'Unassigned';
                    $ticket->type = 'Mail';
                    $ticket->description = $this->extractReplyText($rawBody, $isReply) ?? '';
                    $ticket->attachments = json_encode([]);
                    $ticket->created_by = 1;

                    $ticket->save();
                    event(new CreateTicket($ticket,$ticket));
                    manageCreateTicketPusher($ticket);
                }

                $message->setFlag('SEEN');
            }

            if (! empty($emailRecords)) {
                DB::table('mail2_tickets')->insert($emailRecords);
                Log::info('Inserted '.count($emailRecords).' email records into mail2_tickets.');
            }

            return $emailRecords;
        } catch (Exception $e) {
            Log::error('Error in fetchUnreadEmails: '.$e->getMessage().' | Trace: '.$e->getTraceAsString());

            return [];
        } finally {
            if ($this->client && $this->client->isConnected()) {
                $this->client->disconnect();
                Log::info('Disconnected from IMAP server.');
            }
        }
    }
}
